
import { SelectableField } from '/sd:selectable-field.js'; 
import { ThemeClass } from '/sd:theme-class.js'; 

/**
 * class to create instance of Hoverable Field. It allows for define field which reacts mouse hover ("onmousemove" attribute and "onmouseleave" attribute). The field change class if it is hover.
 */
export class HoverableField extends SelectableField {	
	/**
	 * Constructor
	 * @param    {jQuery} xmlObject    			Data loaded from XML
	 * @param    {String} parentId    			HTML ID of the object, which has to be a parent of constructing object
	 * @param    {String} fieldMainClass    	Name of the main CSS style's class
	 * @param    {String} fieldHoverClass    	Name of the hover CSS style's class
	 * @param	 {String} onMouseMoveAction		This value is insert inside of onmousemove="" attribute of HTML field
	 * @param	 {String} onMouseLeaveAction	This value is insert inside of onmouseleave="" attribute of HTML field
	 * @param	 {String} lmbClickAction		This value is insert inside of onclck="" attribute of HTML field (lmb prefix means Left Mouse Button)
	 * @param	 {String} rmbClickAction		This value is insert inside of oncontextmenu="" attribute of HTML field (rmb prefix means Right Mouse Button)
	 * @param	 {String} selectFunction		This value is insert inside of oncontextmenu="" and onlick="" function supporting unselecting and selecting field
	 * @param    {Integer} itemNumber   		Number of object constructing item - if it is null, it means, that item is not numerable
	 * @param    {String} objId    				ID of creating object - if itemNumber is not null, then itemNumber will be suffix of the ID
	 * @param    {Boolean} selected   			It specifies initial state of the field
	 */
	constructor(xmlObject, parentId, fieldMainClass, fieldHoverClass, onMouseMoveAction = null, onMouseLeaveAction = null, lmbClickAction = null, rmbClickAction = null, selectFunction = null, itemNumber = null, objId = null, selected = false) {
		super(xmlObject, parentId, fieldMainClass, lmbClickAction, rmbClickAction, selectFunction, itemNumber, objId, false);
		this.fieldMainClass = fieldMainClass;
		this.fieldHoverClass = fieldHoverClass;
		if(onMouseMoveAction != null)
			this.mainObj.setAttribute("onmousemove", onMouseMoveAction + ";");
		if(onMouseLeaveAction != null)
			this.mainObj.setAttribute("onmouseleave", onMouseLeaveAction + ";");
	}
	
	/**
	 * Creates object in HTML file - adjusted to work with HoverableField
	 * @override Method in SelectableField
	 * @param {NamedNodeMap} attributes List of attributes of the HTML object loaded from XML
	 */
	createObject(attributes) {
		this.hoverStylesTable = window.theme.getStyleArray(this.id, "hover");
		super.createObject(attributes);
	}

	/**
	 * Changes field's CSS class hover class
	 */
	mouseMoveAction() {
		var element = document.getElementById(this.id);
		element.removeAttribute("style");
		if(this.hoverStylesTable != null) {
			ThemeClass.setTableAsStyle(element, this.hoverStylesTable);
		}
		element.setAttribute("class", this.fieldHoverClass);
	}

	/**
	 * Changes field's CSS class to main class
	 */
	mouseLeaveAction() {
		var element = document.getElementById(this.id);
		this.createThemeSchema();
		element.setAttribute("class", this.fieldMainClass);
	}
}